package ot_const;
use strict;
use vars qw( @ISA @EXPORT_OK $VERSION $testmode $session %ot %valid %r %f %opt );
use Exporter;
@ISA = qw(Exporter);
@EXPORT_OK = qw( %ot %r %f &is &res &is_err %opt $testmode $session );

=head1 NAME

ot_const - Constants for all modules

=head1 SYNOPSIS

This package includes constants for all modules (e.g. error) and some
functions operating on these constants.

=head1 DESCRIPTION

The constants are packed using hashes depending on their use. Functions are
used in order to check, whether a constant property is set or not.

=head2 %ot - general constants

These are general open theory constants. The switch I<$testmode> switches between
test and production. The general constants are:
S<
domain      : domain name
fqdomain    : full qualified name of web domain
maildom     : mail domain name
listmaster  : ot listmaster (email /wo domain)
bb_addr     : identifier for body based mode
sqlhost     : hostname
sqluser     : mysql username
sqlpw       : mysql users password
sqldb       : mysql database name
logpath     : path to logs>

=cut

$testmode = 0;					# switch between test and production
$session = 0;					# session id for protocol set in ot_mail

if( $testmode ){				# run on test system
  %ot = (
 	domain => "opentheory.dom",		# domain name
	fqdomain => "www.opentheory.dom",	# full qualified name of web domain
	maildom => "opentheory.dom",		# mail domain name
	listmaster => "listmaster",		# ot listmaster (email /wo domain)
	bb_addr => "ot",			# identifier for body based mode
	sqlhost => "localhost",			# hostname
	sqluser => "otuser",			# mysql username
	sqlpw => "password",			# mysql users password
	sqldb => "otdb",			# mysql database name
	logpath => "/usr/local/httpd/otlog/"	# path to logs
	);
}
else {						# run on production system
  %ot = (
	domain => "opentheory.org",		# domain name
	fqdomain => "www.opentheory.org",	# full qualified name of web domain
	maildom => "opentheory.org",		# mail domain name
	listmaster => "listmaster",		# ot listmaster (email /wo domain)
	bb_addr => "ot",			# identifier for body based mode
	sqlhost => "localhost",			# hostname
	sqluser => "otuser",			# mysql username
	sqlpw => "password",			# mysql users password
	sqldb => "otdb",			# mysql database name
	logpath => "/data/httpd/opentheory/otlog/"	# path to logs
	);
}

=head2 %valid - command properties, B<%f> - flags for I<%valid> hash

The I<%valid> hash codes the properties of all commands. Properties are
bitwise coded using the flag hash I<%f>:
S<
00000000 00000000 = 0     : invalid command
00000000 00000001 = 1     : comes with projectname
00000000 00000010 = 2     : cmd for ot-members only
00000000 00000100 = 4     : cmd for pj-members only
00000000 00001000 = 8     : cmd for pj-maintainer only
00000000 00010000 = 16    : has no argument
00000000 00100000 = 32    : has one argument
00000000 01000000 = 64    : has two arguments
00000000 10000000 = 128   : has three arguments
00000001 00000000 = 256   : needs confirmation
00000010 00000000 = 512   : cmd for ml-members only
00000100 00000000 = 1024  : cmd for ot-system only
00001000 00000000 = 2048  : command not implemented
00010000 00000000 = 4096  : unused
00100000 00000000 = 8192  : unused
01000000 00000000 = 16384 : unused
10000000 00000000 = 32768 : unused
11111111 11111111 = 65535 : valid command>

NOTE: the I<%valid> hash is an internal structure and can only be accessed
via the funtion I<&is> described below.

The following table shows the coding of the command properties using the
flag hash I<%f>:
S<
Name       | 1 | 2 | 4 | 8 | 16| 32| 64|128|256|512|024|048|
-----------+---+---+---+---+---+---+---+---+---+---+---+---+
which      |   | x |   |   | x |   |   |   |   |   |   |   |
who        | x | x | x |   | x |   |   |   |   |   |   |   |
subscribe  | x |   |   |   | x |   |   |   | x |   |   |   |
unsubscribe| x | x | x |   | x |   |   |   | x |   |   |   |
gettree    | x |   |   |   | x |   |   |   |   |   |   | x |
getlists   | x |   |   |   | x |   |   |   |   |   |   | x |
confirm    |   |   |   |   |   | x |   |   |   |   |   |   |
help       |   |   |   |   |   | x |   |   |   |   |   | x |
submit     | x | x | x | x |   | x |   |   | x |   |   | x |
give       | x | x | x | x |   | x |   |   | x |   |   | x |
addsub     | x | x | x | x |   | x |   |   |   |   |   | x |
uselist    | x | x | x | x |   | x |   |   |   |   |   | x |
keywords   | x | x | x | x |   | x |   |   |   |   |   | x |
title      | x | x | x | x |   | x |   |   |   |   |   | x |
description| x | x | x | x |   | x |   |   |   |   |   | x |
setoption  |   | x |   |   |   |   | x |   |   |   |   |   |
send       |   |   |   |   |   |   |   | x |   |   | x |   |
comment    | x |   |   |   |   |   | x |   |   |   |   | x |
setname    |   | x |   |   |   |   | x |   |   |   |   | x |
newversion | x | x | x | x |   |   | x |   | x |   |   | x |
newproject |   | x | x |   |   |   |   | x | x |   |   | x |
(maillist) | x | x |   |   |   |   | x |   |   | x |   |   |

The command 'maillist' is not set by user, but set by parse_a.>

=cut

%f = (
	invalid_command	=> 0,
	has_pj_name	=> 1,
	ot_members_only	=> 2,
	pj_members_only	=> 4,
	pj_maint_only	=> 8,
	no_arg		=> 16,
	one_arg		=> 32,
	two_args	=> 64,
	three_args	=> 128,
	to_confirm	=> 256,
	ml_members_only	=> 512,
	system_only	=> 1024,
	not_implemented	=> 2048,
	valid_command	=> 65535,
);

%valid = (
	addsub		=> 47+2048,
	comment 	=> 65+2048,
	confirm		=> 32,
	description	=> 47+2048,
	getlists	=> 17+2048,
	gettree		=> 17+2048,
	give		=> 303+2048,
	help		=> 32+2048,
	keywords	=> 47+2048,
	newproject	=> 390+2048,
	newversion	=> 335+2048,
	setname		=> 66+2048,
	setoption	=> 66,
	send		=> 1152,
	submit		=> 303+2048,
	subscribe	=> 273,
	title		=> 47+2048,
	unsubscribe	=> 279,
	uselist		=> 47,
	which		=> 18,
	who		=> 23,
	maillist	=> 579
);

=head2 &is - checks if flag is set

The function I<&is> checks, whether in a command a flag described by I<%f> is set or
not. This means the question, if a command has a property or not.

S<in-param:	$cmd - command
in-param:	$flag - flag coding the command property
returns:	true (flag is set) / false (flag is not set)>

=cut

sub is {
	my( $cmd, $flag ) = @_;
	return((( $valid{$cmd} || $f{invalid_command} ) & $flag ) > 0 );
}

=head2 %r - result codes

All result codes used in all packages are defined by the I<%r> hash. Zero or
negative values indicate errors, positive values indicate normal processing.
However, values can be handled opaque and evaluated by functions I<&is>,
I<&res> and I&<is_err>.

	ok_finished	: processing finished
	ok_confirmed	: normal processing, user confirmed cmd
	ok_bb_mode	: normal processing, body based mode
	ok_ab_mode	: normal processing, address based mode
	ok_nothing_done	: normal processing, but nothing done
	ok		: normal processing
	ok_requested	: normal processing, confirmation is requested
	ok_to_confirm	: normal processing, user has to confirm
	errors		: from here error section starts
	bcc_not_allowed	: bbc-mailing is not allowed
	mixed_mode	: mail mixes ab- and bb-mode
	not_a_command	: addr is not a command (no '*', no 'ot')
	invalid_command	: invalid command
	missing_pjname	: missing project name
	invalid_pjname	: invalid project name
	sh_never_happen	: this error should never happen ;-)
	not_ot_member	: user is not ot-member
	not_pj_member	: user is not project member
	not_maintainer	: user is not project maintainer
	yet_maintainer	: user is yet project maintainer
	new_not_pj_memb	: new maintainer is not project member
	hasn_maillist	: user has not given mailinglist
	nothing_confirm	: there is nothing to confirm
	already_ot_memb	: user is already ot member
	already_pj_memb	: user is already pj member
	invalid_option	: invalid option
	invalid_yesno	: invalid value y(es) or n(o)
	superuser_only	: command is not submited by system/superuser
	invalid_sendopt	: invalid send option
	invalid_sendpjn : invalid send project name
	no_recipients	: no recipients found for mailing
	mail_loop	: mail blocked due to mail loop
	not_implemented	: this command is not yet implemented
	error		: general error, not specified

=cut

%r = (				# list of result codes
	ok_finished	=> 99,
	ok_confirmed	=> 7,
	ok_bb_mode	=> 6,
	ok_ab_mode	=> 5,
	ok_nothing_done	=> 4,
	ok		=> 3,
	ok_requested	=> 2,
	ok_to_confirm	=> 1,
	errors		=> 0,	# codes < errors are "errors"
	bcc_not_allowed	=> -1,
	mixed_mode	=> -2,
	not_a_command	=> -3,
	invalid_command	=> -4,
	missing_pjname	=> -5,
	invalid_pjname	=> -6,
	sh_never_happen	=> -7,
	not_ot_member	=> -8,
	not_pj_member	=> -9,
	not_maintainer	=> -10,
	yet_maintainer	=> -11,
	yet_pj_member	=> -12,
	new_not_pj_memb	=> -13,
	hasnt_maillist	=> -14,
	nothing_confirm	=> -15,
	already_ot_memb	=> -16,
	already_pj_memb	=> -17,
	invalid_option	=> -18,
	invalid_yesno	=> -19,
	superuser_only	=> -20,
	invalid_sendopt	=> -21,
	invalid_sendpjn	=> -22,
	no_recipients	=> -23,
	mail_loop	=> -97,
	not_implemented	=> -98,
	error		=> -99
);

=head2 &is_err - checks, if an error occured

in-param:	$r_val	= result code
returns:	true (an error occured) / false (normal processing)

=cut

sub is_err {
	my( $r_val ) = @_;
	return( $r_val < $r{errors} );
}

=head2 &res - result function, returns key for I<%r> code

in-param:	$r_val	- result code
returns:	$r_key	_ result key

=cut

sub res {
	my( $r_val ) = @_;
	my $r_key = 'key_not_found';	# should never happen;-)
	foreach my $key ( keys %r ){
		if( $r{$key} eq $r_val ) {
			$r_key = $key;
			last;
		}
	}
	return( $r_key );
}

=head2 %opt - ot member options

Members of open theory can un/set options which are coded bitwise and stored
in hash I<%opt> (values are bitshifts). These options are used bei send
command:

	newsletter	= 0	: (=1) open theory newsletter
	newsmaintainer	= 1	: (=2) maintainer newsletter
	newproject	= 2	: (=4) mail if new projects
	newversionall	= 3	: (=8) mail if new versions of all projects
	newversionmy	= 4	: (=16) mail if new versions of subscribed projects

=cut

%opt = (
	newsletter	=> 0,
	newsmaintainer	=> 1,
	newproject	=> 2,
	newversionall	=> 3,
	newversionmy	=> 4
);

1;
